<?php
declare(strict_types=1);

namespace App\Controllers\Settings;

require_once APP_PATH . '/helpers/auth_helper.php';
require_once APP_PATH . '/services/IdentityService.php';
require_once APP_PATH . '/models/LocationIdentity.php';

use \IdentityService;
use \LocationIdentity;


class IdentityController {

  private function guard(): void {
    require_permission('settings.view');
  }

  private function upload_file(string $field, array $allowExt, int $maxBytes): ?string {
    if (empty($_FILES[$field]) || $_FILES[$field]['error'] !== UPLOAD_ERR_OK) return null;

    $tmp  = $_FILES[$field]['tmp_name'];
    $name = $_FILES[$field]['name'];
    $size = (int)$_FILES[$field]['size'];

    if ($size > $maxBytes) throw new \Exception("File {$field} terlalu besar.");

    $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
    if (!in_array($ext, $allowExt, true)) throw new \Exception("Ext {$field} tidak valid.");

    // pakai STORAGE_PATH biar konsisten dengan index.php
    $dir = PUBLIC_PATH . '/storage/uploads';
if (!is_dir($dir)) mkdir($dir, 0775, true);

    $fname = $field . '_' . date('Ymd_His') . '_' . bin2hex(random_bytes(4)) . '.' . $ext;
    $dest  = $dir . '/' . $fname;

    if (!move_uploaded_file($tmp, $dest)) throw new \Exception("Gagal upload {$field}.");

    // simpan path relatif (tanpa domain)
    return 'storage/uploads/' . $fname;
  }

  public function edit() {
    $this->guard();
    $identity = \IdentityService::current();

    // kalau Anda pakai view() helper, ini lebih rapi:
    // view('settings/identity/edit.php', ['identity' => $identity, 'pageTitle' => 'Identitas Lokasi']);
    // tapi kalau mau tetap sesuai file Anda sekarang, biarkan require:
    require APP_PATH . '/views/settings/identity/edit.php';
  }

  public function update() {
    $this->guard();
    $tenant_id   = (int)($_SESSION['tenant_id'] ?? 1);
    $location_id = (int)($_SESSION['location_id'] ?? 1);

    $data = [
      'title'       => trim($_POST['title'] ?? ''),
      'footer_text' => trim($_POST['footer_text'] ?? ''),
    ];

    try {
      $logo_icon = $this->upload_file('logo_icon', ['png','jpg','jpeg','webp'], 2 * 1024 * 1024);
      $logo_text = $this->upload_file('logo_text', ['png','jpg','jpeg','webp'], 2 * 1024 * 1024);
      $favicon   = $this->upload_file('favicon', ['ico','png'], 512 * 1024);

      $current = \LocationIdentity::getByLocation($tenant_id, $location_id) ?: [];

      $data['logo_icon'] = $logo_icon ?: ($current['logo_icon'] ?? null);
      $data['logo_text'] = $logo_text ?: ($current['logo_text'] ?? null);
      $data['favicon']   = $favicon   ?: ($current['favicon'] ?? null);

      \IdentityService::save($tenant_id, $location_id, $data);
      $_SESSION['flash_success'] = "Identitas lokasi berhasil disimpan.";
    } catch (\Exception $e) {
      $_SESSION['flash_error'] = $e->getMessage();
    }

    header("Location: " . url('/settings/identity'));
    exit;
  }
}
