<?php
declare(strict_types=1);

namespace App\Controllers\Settings;

require_once APP_PATH . '/helpers/auth_helper.php';

class RoleController
{
  private function guardView(): void {
    require_permission('settings.roles.view');
  }

  private function guardEdit(): void {
    require_permission('settings.roles.edit');
  }

  private function tenantId(): int {
    return (int)($_SESSION['tenant_id'] ?? 1);
  }

  public function index(): void
  {
    $this->guardView();
    $tenantId = $this->tenantId();

    // roles (system + tenant)
    $stRoles = db()->prepare("
      SELECT id, code, name
      FROM roles
      WHERE tenant_id=? OR tenant_id IS NULL
      ORDER BY id ASC
    ");
    $stRoles->execute([$tenantId]);
    $roles = $stRoles->fetchAll() ?: [];

    // permissions catalog
    $stPerm = db()->prepare("
      SELECT id, code, name, module
      FROM permissions
      ORDER BY module ASC, name ASC
    ");
    $stPerm->execute();
    $permissions = $stPerm->fetchAll() ?: [];

    // map permissions by module for accordion
    $permByModule = [];
    foreach ($permissions as $p) {
      $m = (string)($p['module'] ?? 'OTHER');
      $permByModule[$m][] = $p;
    }

    // load assigned permissions per role
    $rolePerm = []; // [role_id => [perm_code => true]]
    $stRP = db()->prepare("
      SELECT rp.role_id, p.code
      FROM role_permissions rp
      JOIN permissions p ON p.id = rp.permission_id
    ");
    $stRP->execute();
    foreach (($stRP->fetchAll() ?: []) as $r) {
      $rid = (int)$r['role_id'];
      $code = (string)$r['code'];
      if ($code !== '') $rolePerm[$rid][$code] = true;
    }

    require APP_PATH . '/views/settings/roles/index.php';
  }

  public function store(): void
  {
    $this->guardEdit();
    $tenantId = $this->tenantId();

    $code = trim((string)($_POST['code'] ?? ''));
    $name = trim((string)($_POST['name'] ?? ''));
    $permCodes = (array)($_POST['permissions'] ?? []);

    if ($code === '' || $name === '') {
      $_SESSION['flash_error'] = 'Kode dan nama role wajib diisi.';
      header('Location: ' . url('/settings/roles'));
      exit;
    }

    // validate code format (lowercase, underscore, digits, 3-32)
    if (!preg_match('/^[a-z0-9_]{3,32}$/', $code)) {
      $_SESSION['flash_error'] = 'Kode role tidak valid (a-z 0-9 underscore, 3-32).';
      header('Location: ' . url('/settings/roles'));
      exit;
    }

    try {
      db()->beginTransaction();

      // insert role
      $st = db()->prepare("INSERT INTO roles (tenant_id, code, name) VALUES (?, ?, ?)");
      $st->execute([$tenantId, $code, $name]);
      $roleId = (int)db()->lastInsertId();

      $this->syncPermissions($roleId, $permCodes);

      db()->commit();
      $_SESSION['flash_success'] = 'Role berhasil dibuat.';
      header('Location: ' . url('/settings/roles'));
      exit;
    } catch (\Throwable $e) {
      if (db()->inTransaction()) db()->rollBack();
      $_SESSION['flash_error'] = 'Gagal membuat role: ' . $e->getMessage();
      header('Location: ' . url('/settings/roles'));
      exit;
    }
  }

  public function update(): void
  {
    $this->guardEdit();
    $tenantId = $this->tenantId();

    $roleId = (int)($_POST['role_id'] ?? 0);
    $name = trim((string)($_POST['name'] ?? ''));
    $permCodes = (array)($_POST['permissions'] ?? []);

    if ($roleId <= 0 || $name === '') {
      $_SESSION['flash_error'] = 'Data role tidak valid.';
      header('Location: ' . url('/settings/roles'));
      exit;
    }

    try {
      db()->beginTransaction();

      // ensure role belongs to tenant (or system)
      $stCheck = db()->prepare("SELECT id, tenant_id, code FROM roles WHERE id=? LIMIT 1");
      $stCheck->execute([$roleId]);
      $role = $stCheck->fetch();
      if (!$role) {
        db()->rollBack();
        $_SESSION['flash_error'] = 'Role tidak ditemukan.';
        header('Location: ' . url('/settings/roles'));
        exit;
      }

      // prevent edit system owner/admin roles? (optional)
      // if ((string)$role['code'] === 'owner') { ... }

      // update name (only if tenant role or allowed)
      $stUpd = db()->prepare("UPDATE roles SET name=? WHERE id=?");
      $stUpd->execute([$name, $roleId]);

      $this->syncPermissions($roleId, $permCodes);

      db()->commit();

      // important: role/permission changed → clear cache current user
      clear_permission_cache();

      $_SESSION['flash_success'] = 'Role berhasil disimpan.';
      header('Location: ' . url('/settings/roles'));
      exit;
    } catch (\Throwable $e) {
      if (db()->inTransaction()) db()->rollBack();
      $_SESSION['flash_error'] = 'Gagal simpan role: ' . $e->getMessage();
      header('Location: ' . url('/settings/roles'));
      exit;
    }
  }

  private function syncPermissions(int $roleId, array $permCodes): void
  {
    // normalize codes
    $permCodes = array_values(array_unique(array_filter(array_map('strval', $permCodes))));

    // map codes → permission_id
    if (count($permCodes) > 0) {
      $in = implode(',', array_fill(0, count($permCodes), '?'));
      $st = db()->prepare("SELECT id, code FROM permissions WHERE code IN ($in)");
      $st->execute($permCodes);
      $rows = $st->fetchAll() ?: [];
      $ids = [];
      foreach ($rows as $r) $ids[] = (int)$r['id'];
    } else {
      $ids = [];
    }

    // reset
    db()->prepare("DELETE FROM role_permissions WHERE role_id=?")->execute([$roleId]);

    // insert
    if (count($ids) > 0) {
      $ins = db()->prepare("INSERT INTO role_permissions (role_id, permission_id) VALUES (?, ?)");
      foreach ($ids as $pid) $ins->execute([$roleId, $pid]);
    }
  }
}
