<?php
declare(strict_types=1);

namespace App\Controllers\Settings;

require_once APP_PATH . '/helpers/url_helper.php';
require_once APP_PATH . '/helpers/db.php';
require_once APP_PATH . '/helpers/auth_helper.php';
require_once APP_PATH . '/models/Location.php';



class UserController
{
  private function tenantId(): int {
    return (int)($_SESSION['tenant_id'] ?? 1);
  }

  private function requireUsersAccess(): void {
    // Sprint 2 Patch 2: minimal gate
    require_permission('settings.users');
  }

  public function index(): void
  {
    $this->requireUsersAccess();

    $tenantId = $this->tenantId();

    $sql = "
      SELECT
        u.id, u.name, u.email, u.thumbnail, u.is_active,
        GROUP_CONCAT(DISTINCT r.name ORDER BY r.name SEPARATOR ', ') AS roles_text,
        COUNT(DISTINCT ul.location_id) AS locations_count
      FROM users u
      LEFT JOIN user_roles ur ON ur.user_id = u.id
      LEFT JOIN roles r ON r.id = ur.role_id
      LEFT JOIN user_locations ul ON ul.user_id = u.id
      WHERE u.tenant_id = ?
      GROUP BY u.id
      ORDER BY u.id DESC
    ";
    $st = db()->prepare($sql);
    $st->execute([$tenantId]);
    $users = $st->fetchAll();

    $pageTitle = 'Users';
    $breadcrumb = [
  ['label' => 'Settings', 'href' => url('/settings')],
  ['label' => 'Users', 'href' => null],
];


    require APP_PATH . '/views/settings/users/index.php';
  }
  
  private function pdo()
{
    global $pdo;
    return $pdo;
}


public function create(): void
{
  $this->requireUsersAccess();

  $tenantId = $this->tenantId();

  $rolesSt = db()->prepare('SELECT id, name, code FROM roles WHERE tenant_id=? ORDER BY is_system DESC, name ASC');
  $rolesSt->execute([$tenantId]);
  $roles = $rolesSt->fetchAll();

  $locations = \Location::allByTenant($tenantId);

  $pageTitle = 'Tambah User';
  $breadcrumb = [
    ['label' => 'Settings', 'url' => url('/settings')],
    ['label' => 'Users', 'url' => url('/settings/users')],
    ['label' => 'Tambah User'],
  ];

  require APP_PATH . '/views/settings/users/create.php';
}

  public function store(): void
{
  $this->requireUsersAccess();
  $tenantId = $this->tenantId();

  $name      = trim((string)($_POST['name'] ?? ''));
  $email     = trim((string)($_POST['email'] ?? ''));
  $jobTitle  = trim((string)($_POST['job_title'] ?? ''));
  $password  = (string)($_POST['password'] ?? '');
  $password2 = (string)($_POST['password_confirm'] ?? '');

  $roleIds     = array_values(array_unique(array_map('intval', (array)($_POST['roles'] ?? []))));
  $locationIds = array_values(array_unique(array_map('intval', (array)($_POST['locations'] ?? []))));

  // ===== Basic validation =====
  if ($name === '' || $email === '' || $password === '') {
    $_SESSION['flash_error'] = 'Nama, email, dan password wajib diisi.';
    header('Location: ' . url('/settings/users/create'));
    exit;
  }
  if ($password !== $password2) {
    $_SESSION['flash_error'] = 'Konfirmasi password tidak sama.';
    header('Location: ' . url('/settings/users/create'));
    exit;
  }
  if (empty($roleIds)) {
    $_SESSION['flash_error'] = 'Minimal pilih 1 role.';
    header('Location: ' . url('/settings/users/create'));
    exit;
  }
  if (empty($locationIds)) {
    $_SESSION['flash_error'] = 'Minimal pilih 1 lokasi.';
    header('Location: ' . url('/settings/users/create'));
    exit;
  }

  // ===== Upload thumbnail (optional) =====
  $thumbPath = null;
  if (!empty($_FILES['thumbnail']['name'] ?? '')) {
    $dir = PUBLIC_PATH . '/storage/uploads/users';

    if (!is_dir($dir)) @mkdir($dir, 0775, true);

    $ext = strtolower(pathinfo((string)$_FILES['thumbnail']['name'], PATHINFO_EXTENSION));
    $allowed = ['png','jpg','jpeg','webp'];

    if (!in_array($ext, $allowed, true)) {
      $_SESSION['flash_error'] = 'Format thumbnail tidak didukung.';
      header('Location: ' . url('/settings/users/create'));
      exit;
    }

    $fn = 'u_' . time() . '_' . bin2hex(random_bytes(4)) . '.' . $ext;
    $target = $dir . '/' . $fn;

    if (!move_uploaded_file((string)($_FILES['thumbnail']['tmp_name'] ?? ''), $target)) {
      $_SESSION['flash_error'] = 'Gagal upload thumbnail.';
      header('Location: ' . url('/settings/users/create'));
      exit;
    }

    // asset_url() baca dari /storage
    $thumbPath = 'storage/uploads/users/' . $fn;
  }

  $hash = password_hash($password, PASSWORD_DEFAULT);

  try {
    db()->beginTransaction();

    // ===== Prevent duplicate email in tenant =====
    $stDup = db()->prepare('SELECT id FROM users WHERE tenant_id=? AND email=? LIMIT 1');
    $stDup->execute([$tenantId, $email]);
    if ($stDup->fetch()) {
      db()->rollBack();
      $_SESSION['flash_error'] = 'Email sudah dipakai.';
      header('Location: ' . url('/settings/users/create'));
      exit;
    }

    // ===== Validate roles belong to tenant =====
    $inRoles = implode(',', array_fill(0, count($roleIds), '?'));
    $stRoleOk = db()->prepare("SELECT id FROM roles WHERE tenant_id=? AND id IN ($inRoles)");
    $stRoleOk->execute(array_merge([$tenantId], $roleIds));
    $validRoleIds = array_map(fn($r) => (int)$r['id'], $stRoleOk->fetchAll() ?: []);
    if (count($validRoleIds) !== count($roleIds)) {
      db()->rollBack();
      $_SESSION['flash_error'] = 'Role tidak valid.';
      header('Location: ' . url('/settings/users/create'));
      exit;
    }

    // ===== Validate locations belong to tenant =====
    $inLocs = implode(',', array_fill(0, count($locationIds), '?'));
    $stLocOk = db()->prepare("SELECT id FROM locations WHERE tenant_id=? AND id IN ($inLocs)");
    $stLocOk->execute(array_merge([$tenantId], $locationIds));
    $validLocIds = array_map(fn($r) => (int)$r['id'], $stLocOk->fetchAll() ?: []);
    if (count($validLocIds) !== count($locationIds)) {
      db()->rollBack();
      $_SESSION['flash_error'] = 'Lokasi tidak valid.';
      header('Location: ' . url('/settings/users/create'));
      exit;
    }

    // ===== Insert user =====
    if ($this->usersHasJobTitle()) {
      $st = db()->prepare('
        INSERT INTO users (tenant_id, name, email, job_title, password_hash, thumbnail, is_active, is_owner, created_at)
        VALUES (?, ?, ?, ?, ?, ?, 1, 0, NOW())
      ');
      $st->execute([$tenantId, $name, $email, $jobTitle, $hash, $thumbPath]);
    } else {
      $st = db()->prepare('
        INSERT INTO users (tenant_id, name, email, password_hash, thumbnail, is_active, is_owner, created_at)
        VALUES (?, ?, ?, ?, ?, 1, 0, NOW())
      ');
      $st->execute([$tenantId, $name, $email, $hash, $thumbPath]);
    }

    $userId = (int)db()->lastInsertId();

    // ===== Insert roles =====
    $insRole = db()->prepare('INSERT INTO user_roles (user_id, role_id) VALUES (?, ?)');
    foreach ($roleIds as $rid) {
      if ($rid > 0) $insRole->execute([$userId, $rid]);
    }

    // ===== Insert locations =====
    $insLoc = db()->prepare('INSERT INTO user_locations (user_id, location_id) VALUES (?, ?)');
    foreach ($locationIds as $lid) {
      if ($lid > 0) $insLoc->execute([$userId, $lid]);
    }

    db()->commit();

    $_SESSION['flash_success'] = 'User berhasil dibuat.';
    header('Location: ' . url('/settings/users'));
    exit;

  } catch (\Throwable $e) {
    if (db()->inTransaction()) db()->rollBack();
    $_SESSION['flash_error'] = 'Gagal membuat user: ' . $e->getMessage();
    header('Location: ' . url('/settings/users/create'));
    exit;
  }
}

 public function edit(): void
{
  $this->requireUsersAccess();

  $tenantId = $this->tenantId();
  $id = (int)($_GET['id'] ?? 0);
  if ($id <= 0) {
    http_response_code(400);
    echo 'Bad Request: missing id';
    return;
  }

  // user
  $st = db()->prepare("SELECT * FROM users WHERE id=? AND tenant_id=? LIMIT 1");
  $st->execute([$id, $tenantId]);
  $user = $st->fetch();
  if (!$user) {
    http_response_code(404);
    echo 'User not found';
    return;
  }

  // semua roles
  $rolesSt = db()->prepare("SELECT id, name, code FROM roles WHERE tenant_id=? ORDER BY is_system DESC, name ASC");
  $rolesSt->execute([$tenantId]);
  $roles = $rolesSt->fetchAll();

  // roles user
  $urSt = db()->prepare("SELECT role_id FROM user_roles WHERE user_id=?");
  $urSt->execute([$id]);
  $userRoleIds = array_map(fn($r) => (int)$r['role_id'], $urSt->fetchAll() ?: []);

  // semua lokasi (selaras dengan create)
  $locations = \Location::allByTenant($tenantId);

  // lokasi user
  $ulSt = db()->prepare("SELECT location_id FROM user_locations WHERE user_id=?");
  $ulSt->execute([$id]);
  $userLocationIds = array_map(fn($r) => (int)$r['location_id'], $ulSt->fetchAll() ?: []);

  $pageTitle = 'Edit User';
  $breadcrumb = [
    ['label' => 'Settings', 'url' => url('/settings')],
    ['label' => 'Users', 'url' => url('/settings/users')],
    ['label' => 'Edit User'],
  ];

  require APP_PATH . '/views/settings/users/edit.php';
}


  public function update(): void
{
  $this->requireUsersAccess();
  $tenantId = $this->tenantId();

  $id         = (int)($_POST['id'] ?? 0);
  $name       = trim((string)($_POST['name'] ?? ''));
  $email      = trim((string)($_POST['email'] ?? ''));
  $jobTitle   = trim((string)($_POST['job_title'] ?? ''));
  $password   = (string)($_POST['password'] ?? '');
  $password2  = (string)($_POST['password_confirm'] ?? '');
  $roleIds    = array_values(array_unique(array_map('intval', (array)($_POST['roles'] ?? []))));
  $locationIds= array_values(array_unique(array_map('intval', (array)($_POST['locations'] ?? []))));

  // ===== Basic validation =====
  if ($id <= 0) {
    header('Location: ' . url('/settings/users'));
    exit;
  }
  if ($name === '' || $email === '') {
    $_SESSION['flash_error'] = 'Nama dan email wajib diisi.';
    header('Location: ' . url('/settings/users/edit?id=' . $id));
    exit;
  }
  if ($password !== '' && $password !== $password2) {
    $_SESSION['flash_error'] = 'Konfirmasi password tidak sama.';
    header('Location: ' . url('/settings/users/edit?id=' . $id));
    exit;
  }
  if (empty($roleIds)) {
    $_SESSION['flash_error'] = 'Minimal pilih 1 role.';
    header('Location: ' . url('/settings/users/edit?id=' . $id));
    exit;
  }
  if (empty($locationIds)) {
    $_SESSION['flash_error'] = 'Minimal pilih 1 lokasi.';
    header('Location: ' . url('/settings/users/edit?id=' . $id));
    exit;
  }

  // ===== Upload thumbnail (optional) =====
  $thumbPath = null;
  if (!empty($_FILES['thumbnail']['name'] ?? '')) {
    $dir = PUBLIC_PATH . '/storage/uploads/users';
    if (!is_dir($dir)) @mkdir($dir, 0775, true);

    $ext = strtolower(pathinfo((string)$_FILES['thumbnail']['name'], PATHINFO_EXTENSION));
    $allowed = ['png', 'jpg', 'jpeg', 'webp'];

    if (!in_array($ext, $allowed, true)) {
      $_SESSION['flash_error'] = 'Format thumbnail tidak didukung.';
      header('Location: ' . url('/settings/users/edit?id=' . $id));
      exit;
    }

    $fn = 'u_' . time() . '_' . bin2hex(random_bytes(4)) . '.' . $ext;
    $target = $dir . '/' . $fn;

    if (!move_uploaded_file((string)($_FILES['thumbnail']['tmp_name'] ?? ''), $target)) {
      $_SESSION['flash_error'] = 'Gagal upload thumbnail.';
      header('Location: ' . url('/settings/users/edit?id=' . $id));
      exit;
    }

    $thumbPath = 'storage/uploads/users/' . $fn;
  }

  try {
    db()->beginTransaction();

    // ===== Ensure user exists in this tenant =====
    $stUser = db()->prepare('SELECT id, is_owner FROM users WHERE tenant_id=? AND id=? LIMIT 1');
    $stUser->execute([$tenantId, $id]);
    $userRow = $stUser->fetch();
    if (!$userRow) {
      db()->rollBack();
      $_SESSION['flash_error'] = 'User tidak ditemukan.';
      header('Location: ' . url('/settings/users'));
      exit;
    }

    // ===== Validate roles belong to tenant =====
    $inRoles = implode(',', array_fill(0, count($roleIds), '?'));
    $stRoleOk = db()->prepare("SELECT id FROM roles WHERE tenant_id=? AND id IN ($inRoles)");
    $stRoleOk->execute(array_merge([$tenantId], $roleIds));
    $validRoleIds = array_map(fn($r) => (int)$r['id'], $stRoleOk->fetchAll() ?: []);
    if (count($validRoleIds) !== count($roleIds)) {
      db()->rollBack();
      $_SESSION['flash_error'] = 'Role tidak valid.';
      header('Location: ' . url('/settings/users/edit?id=' . $id));
      exit;
    }

    // ===== Validate locations belong to tenant =====
    $inLocs = implode(',', array_fill(0, count($locationIds), '?'));
    $stLocOk = db()->prepare("SELECT id FROM locations WHERE tenant_id=? AND id IN ($inLocs)");
    $stLocOk->execute(array_merge([$tenantId], $locationIds));
    $validLocIds = array_map(fn($r) => (int)$r['id'], $stLocOk->fetchAll() ?: []);
    if (count($validLocIds) !== count($locationIds)) {
      db()->rollBack();
      $_SESSION['flash_error'] = 'Lokasi tidak valid.';
      header('Location: ' . url('/settings/users/edit?id=' . $id));
      exit;
    }

    // ===== Build UPDATE users =====
    $fields = [
      'name'      => $name,
      'email'     => $email,
      'tenant_id' => $tenantId,
      'id'        => $id,
    ];

    $sql = 'UPDATE users SET name=:name, email=:email';

    if ($this->usersHasJobTitle()) {
      $sql .= ', job_title=:job_title';
      $fields['job_title'] = $jobTitle;
    }

    if ($thumbPath !== null) {
      $sql .= ', thumbnail=:thumbnail';
      $fields['thumbnail'] = $thumbPath;
    }

    if ($password !== '') {
      $sql .= ', password_hash=:password_hash';
      $fields['password_hash'] = password_hash($password, PASSWORD_DEFAULT);
    }

    $sql .= ', updated_at=NOW() WHERE tenant_id=:tenant_id AND id=:id';

    $stUpd = db()->prepare($sql);
    $stUpd->execute($fields);

    // ===== Reset roles =====
    db()->prepare('DELETE FROM user_roles WHERE user_id=?')->execute([$id]);
    $insRole = db()->prepare('INSERT INTO user_roles (user_id, role_id) VALUES (?, ?)');
    foreach ($roleIds as $rid) {
      if ($rid > 0) $insRole->execute([$id, $rid]);
    }

    // ===== Reset locations =====
    db()->prepare('DELETE FROM user_locations WHERE user_id=?')->execute([$id]);
    $insLoc = db()->prepare('INSERT INTO user_locations (user_id, location_id) VALUES (?, ?)');
    foreach ($locationIds as $lid) {
      if ($lid > 0) $insLoc->execute([$id, $lid]);
    }

    db()->commit();

    // ===== Refresh session if editing self (sidebar thumb & name live) =====
    if ((int)($_SESSION['user_id'] ?? 0) === $id) {
      $_SESSION['user_name']  = $name;
      $_SESSION['user_email'] = $email;
      if ($thumbPath !== null) {
        $_SESSION['user_thumb'] = $thumbPath;
      }
    }

    $_SESSION['flash_success'] = 'User berhasil diperbarui.';
    header('Location: ' . url('/settings/users/edit?id=' . $id));
    exit;

  } catch (\Throwable $e) {
    if (db()->inTransaction()) db()->rollBack();
    $_SESSION['flash_error'] = 'Gagal update user: ' . $e->getMessage();
    header('Location: ' . url('/settings/users/edit?id=' . $id));
    exit;
  }
}


  public function toggle(): void
  {
    $this->requireUsersAccess();

    $tenantId = $this->tenantId();
    $id = (int)($_POST['id'] ?? 0);
    $active = (int)($_POST['is_active'] ?? 1);

    if ($id <= 0) {
      header('Location: ' . url('/settings/users'));
      exit;
    }

    // Jangan nonaktifkan diri sendiri (safety kecil)
    if ($id === (int)($_SESSION['user_id'] ?? 0) && $active === 0) {
      $_SESSION['flash_error'] = 'Tidak boleh menonaktifkan akun yang sedang dipakai.';
      header('Location: ' . url('/settings/users'));
      exit;
    }

    $st = db()->prepare('UPDATE users SET is_active=?, updated_at=NOW() WHERE tenant_id=? AND id=?');
    $st->execute([$active, $tenantId, $id]);

    $_SESSION['flash_success'] = 'Status user tersimpan.';
    header('Location: ' . url('/settings/users'));
    exit;
  }
  
  private static ?bool $usersHasJobTitle = null;

private function usersHasJobTitle(): bool
{
  if (self::$usersHasJobTitle !== null) return self::$usersHasJobTitle;

  $st = db()->prepare("
    SELECT COUNT(*) c
    FROM information_schema.COLUMNS
    WHERE TABLE_SCHEMA = DATABASE()
      AND TABLE_NAME = 'users'
      AND COLUMN_NAME = 'job_title'
  ");
  $st->execute();
  $row = $st->fetch();

  self::$usersHasJobTitle = ((int)($row['c'] ?? 0) > 0);
  return self::$usersHasJobTitle;
}

}
