<?php
declare(strict_types=1);

/**
 * AUTH + PERMISSION HELPERS (Sprint 2)
 * - Owner bypass
 * - Permission cache in session
 * - Friendly 403/404 page (layout + alert)
 *
 * Tables:
 * - users(id, tenant_id, is_owner, ...)
 * - roles(id, code, ...)
 * - permissions(id, code, name, module, created_at)
 * - user_roles(user_id, role_id)
 * - role_permissions(role_id, permission_id)
 */

/**
 * Guard: prevent redeclare when this file is included more than once.
 */
if (defined('RAJAWALI_AUTH_HELPER_LOADED')) {
  return;
}
define('RAJAWALI_AUTH_HELPER_LOADED', true);

/** Ensure session */
if (session_status() !== PHP_SESSION_ACTIVE) {
  @session_start();
}

/**
 * Soft dependencies:
 * - db() helper (PDO)
 * - url() helper (optional)
 * - asset_url() helper (optional)
 *
 * We won't redeclare them if already exist.
 */
if (!function_exists('db')) {
  function db(): \PDO {
    throw new \RuntimeException('db() helper not available. Pastikan db() sudah didefinisikan sebelum auth_helper.php dipakai.');
  }
}

if (!function_exists('url')) {
  function url(string $path = '/'): string {
    // fallback minimal (biar tidak fatal)
    return $path;
  }
}

/* =========================================================
   AUTH SESSION HELPERS
   ========================================================= */

if (!function_exists('auth_user_id')) {
  function auth_user_id(): int {
    return (int)($_SESSION['user_id'] ?? 0);
  }
}

if (!function_exists('auth_tenant_id')) {
  function auth_tenant_id(): int {
    return (int)($_SESSION['tenant_id'] ?? 0);
  }
}

if (!function_exists('auth_is_owner')) {
  function auth_is_owner(): bool {
    return ((int)($_SESSION['is_owner'] ?? 0) === 1);
  }
}

if (!function_exists('require_auth')) {
  function require_auth(): void {
    if (auth_user_id() <= 0) {
      header('Location: ' . url('/login'));
      exit;
    }
  }
}

/**
 * Call this right after successful login.
 * (Optional helper, biar konsisten)
 */
if (!function_exists('auth_set_session_from_user_row')) {
  function auth_set_session_from_user_row(array $user): void {
    $_SESSION['user_id']   = (int)($user['id'] ?? 0);
    $_SESSION['tenant_id'] = (int)($user['tenant_id'] ?? 0);
    $_SESSION['is_owner']  = (int)($user['is_owner'] ?? 0);

    // UI convenience
    if (isset($user['name']))  $_SESSION['user_name']  = (string)$user['name'];
    if (isset($user['email'])) $_SESSION['user_email'] = (string)$user['email'];
    if (isset($user['thumbnail'])) $_SESSION['user_thumb'] = (string)$user['thumbnail'];

    // permission cache harus reset setelah login
    if (function_exists('clear_permission_cache')) {
      clear_permission_cache();
    }
  }
}

/* =========================================================
   PERMISSION CACHE
   ========================================================= */

if (!function_exists('_perm_cache_session_key')) {
  function _perm_cache_session_key(): string {
    // cache per tenant + per user
    return 'perm_codes:' . auth_tenant_id() . ':' . auth_user_id();
  }
}

if (!function_exists('clear_permission_cache')) {
  function clear_permission_cache(): void {
    unset($_SESSION[_perm_cache_session_key()]);
  }
}

if (!function_exists('_load_permission_codes_from_db')) {
  function _load_permission_codes_from_db(): array
  {
    if (auth_is_owner()) return ['*'];

    $uid = auth_user_id();
    $tid = auth_tenant_id();
    if ($uid <= 0 || $tid <= 0) return [];

    $sql = "
      SELECT DISTINCT p.code
      FROM users u
      JOIN user_roles ur ON ur.user_id = u.id
      JOIN role_permissions rp ON rp.role_id = ur.role_id
      JOIN permissions p ON p.id = rp.permission_id
      WHERE u.id = ? AND u.tenant_id = ?
    ";
    $st = db()->prepare($sql);
    $st->execute([$uid, $tid]);

    $rows = $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
    $codes = [];
    foreach ($rows as $r) {
      $c = (string)($r['code'] ?? '');
      if ($c !== '') $codes[$c] = true;
    }
    return array_keys($codes);
  }
}

if (!function_exists('permission_codes')) {
  function permission_codes(): array
  {
    $key = _perm_cache_session_key();
    if (!isset($_SESSION[$key]) || !is_array($_SESSION[$key])) {
      $_SESSION[$key] = _load_permission_codes_from_db();
    }
    return $_SESSION[$key];
  }
}

/* =========================================================
   PERMISSION API
   ========================================================= */

if (!function_exists('has_permission')) {
  function has_permission(string $code): bool
  {
    if ($code === '') return false;

    if (auth_is_owner()) return true;

    $codes = permission_codes();
    if (in_array('*', $codes, true)) return true;

    return in_array($code, $codes, true);
  }
}

if (!function_exists('require_permission')) {
  function require_permission(string $code): void
  {
    require_auth();

    if (has_permission($code)) return;

    // Friendly page, not raw "403 Forbidden"
    $_SESSION['flash_error'] = 'Akses ditolak. Anda tidak memiliki izin untuk membuka halaman ini.';
    forbidden_page();
  }
}

if (!function_exists('require_any_permission')) {
  function require_any_permission(array $codes): void
  {
    require_auth();

    if (auth_is_owner()) return;

    foreach ($codes as $c) {
      $c = (string)$c;
      if ($c !== '' && has_permission($c)) return;
    }

    $_SESSION['flash_error'] = 'Akses ditolak. Anda tidak memiliki izin untuk membuka halaman ini.';
    forbidden_page();
  }
}

/* =========================================================
   FRIENDLY ERROR PAGES (layout + empty content)
   ========================================================= */

/**
 * Render a minimal "empty page" inside your existing layout.
 * Adjust paths if your layout is different.
 *
 * Project kamu punya: views/layouts/header.php, sidebar.php, footer.php
 */
if (!function_exists('_render_empty_page')) {
  function _render_empty_page(string $title, string $message, int $httpCode = 200): void
  {
    // We return 200 to avoid browser "error page" / avoid scary 403/404.
    // If you want real 403/404 later, change to $httpCode.
    http_response_code($httpCode);

    // minimal variables commonly used by layout
    $pageTitle = $title;

    // basic content
    $msg = $message;

    // Try to use your standard layout (header/sidebar/footer)
    $header = APP_PATH . '/views/layouts/header.php';
    $sidebar = APP_PATH . '/views/layouts/sidebar.php';
    $footer = APP_PATH . '/views/layouts/footer.php';

    if (is_file($header)) require $header;
    if (is_file($sidebar)) require $sidebar;

    // Content area (simple, clean)
    echo '<main class="page-content" style="padding:24px;">';
    echo '  <div class="card" style="padding:18px;border-radius:14px;">';
    echo '    <div class="alert alert--danger" style="margin:0;">' . htmlspecialchars($msg) . '</div>';
    echo '  </div>';
    echo '</main>';

    if (is_file($footer)) require $footer;

    exit;
  }
}

if (!function_exists('forbidden_page')) {
  function forbidden_page(): void
  {
    $msg = (string)($_SESSION['flash_error'] ?? 'Akses ditolak.');
    unset($_SESSION['flash_error']);
    _render_empty_page('Akses Ditolak', $msg, 200);
  }
}

if (!function_exists('not_found_page')) {
  function not_found_page(string $customMessage = ''): void
  {
    $msg = $customMessage !== '' ? $customMessage : 'Halaman tidak ditemukan.';
    _render_empty_page('Halaman Tidak Ditemukan', $msg, 200);
  }
}
